<?php
/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2023 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */

namespace App\Process\Service\RecordThreadListActions;

use ApiPlatform\Core\Exception\InvalidArgumentException;
use App\Data\LegacyHandler\PresetDataHandlers\AlertsDataHandler;
use App\Data\Service\RecordMarkAsReadServiceInterface;
use App\Engine\LegacyHandler\LegacyHandler;
use App\Engine\LegacyHandler\LegacyScopeState;
use App\Module\Service\ModuleNameMapperInterface;
use App\Process\Entity\Process;
use App\Process\Service\ProcessHandlerInterface;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

class MarkAsReadRecordThreadListAction extends LegacyHandler implements ProcessHandlerInterface
{
    protected const MSG_OPTIONS_NOT_FOUND = 'Process options are not defined';

    protected const PROCESS_TYPE = 'record-thread-list-mark-as-read';

    /**
     * @var RecordMarkAsReadServiceInterface
     */
    protected $recordMarkAsReadService;

    /**
     * @var AlertsDataHandler
     */
    protected $alertsDataHandler;

    /**
     * @var ModuleNameMapperInterface
     */
    protected $moduleNameMapper;

    /**
     * MarkAsReadRecordThreadListAction constructor.
     * @param string $projectDir
     * @param string $legacyDir
     * @param string $legacySessionName
     * @param string $defaultSessionName
     * @param LegacyScopeState $legacyScopeState
     * @param SessionInterface $session
     * @param RecordMarkAsReadServiceInterface $recordMarkAsReadService
     * @param AlertsDataHandler $alertsDataHandler
     * @param ModuleNameMapperInterface $moduleNameMapper
     */
    public function __construct(
        string $projectDir,
        string $legacyDir,
        string $legacySessionName,
        string $defaultSessionName,
        LegacyScopeState $legacyScopeState,
        SessionInterface $session,
        RecordMarkAsReadServiceInterface $recordMarkAsReadService,
        AlertsDataHandler $alertsDataHandler,
        ModuleNameMapperInterface $moduleNameMapper
    )
    {
        parent::__construct($projectDir, $legacyDir, $legacySessionName, $defaultSessionName, $legacyScopeState, $session);
        $this->recordMarkAsReadService = $recordMarkAsReadService;
        $this->alertsDataHandler = $alertsDataHandler;
        $this->moduleNameMapper = $moduleNameMapper;
    }

    /**
     * {@inheritDoc}
     */
    public function getHandlerKey(): string
    {
        return self::PROCESS_TYPE;
    }

    /**
     * {@inheritDoc}
     */
    public function getProcessType(): string
    {
        return self::PROCESS_TYPE;
    }

    /**
     * {@inheritDoc}
     */
    public function requiredAuthRole(): string
    {
        return 'ROLE_USER';
    }

    /**
     * @inheritDoc
     */
    public function getRequiredACLs(Process $process): array
    {
        $options = $process->getOptions();
        $module = $options['module'] ?? '';
        $ids = $options['ids'] ?? [];

        return [
            $module => [
                [
                    'action' => 'view',
                    'ids' => $ids
                ]
            ],
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function configure(Process $process): void
    {
        //This process is synchronous
        //We aren't going to store a record on db
        //thus we will use process type as the id
        $process->setId(self::PROCESS_TYPE);
        $process->setAsync(false);
    }

    /**
     * {@inheritDoc}
     */
    public function validate(Process $process): void
    {
        if (empty($process->getOptions())) {
            throw new InvalidArgumentException(self::MSG_OPTIONS_NOT_FOUND);
        }

        $options = $process->getOptions();

        if (empty($options['module']) || empty($options['action'])) {
            throw new InvalidArgumentException(self::MSG_OPTIONS_NOT_FOUND);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function run(Process $process)
    {
        $options = $process->getOptions();

        if (empty($options['ids'])) {
            $process->setStatus('success');
            $process->setMessages([]);

            return;
        }

        $result = $this->markAllAsRead($process);

        $process->setStatus('success');

        if (!$result) {
            $process->setStatus('error');
            $process->setMessages(['ERR_NOTIFICATIONS_MARK_AS_READ']);

            return;
        }

        $legacyModuleName = $this->moduleNameMapper->toLegacy($process->getOptions()['module'] ?? '');

        $this->init();
        $unreadCount = $this->alertsDataHandler->getUnreadCount($legacyModuleName);
        $this->close();

        $responseData = [
            'reloadThread' => true,
            'unreadCount' => $unreadCount
        ];

        $process->setData($responseData);
    }

    protected function markAllAsRead(Process $process): bool
    {
        $options = $process->getOptions();

        return $this->recordMarkAsReadService->markRecordsAsRead($options['module'], $options['ids'] ?? []);

    }
}
